<#
.SYNOPSIS
    Scans Program Files directories on a source drive for critical config/license files 
    and mirrors them to a backup location using Robocopy, preserving directory structure.

.DESCRIPTION
    1. Scans "Program Files" and "Program Files (x86)" on the source drive.
    2. Looks for specific extensions (.ini, .lic, .db, etc.).
    3. Calculates the destination path by swapping the drive root.
    4. Uses Robocopy to copy the file (and its attributes/timestamps).
#>

# --- CONFIGURATION ---
$SourceDrive = "D:"
$BackupRoot  = "C:\shared\share\backups\rtp\lynch"

# The folders to scan on the Source Drive
$ScanPaths = @(
    "$SourceDrive\Program Files", 
    "$SourceDrive\Program Files (x86)"
)

# The extensions to look for
$Extensions = @(
    "*.ini", "*.conf", "*.cfg", "*.lic", "*.key", 
    "*.mdb", "*.db", "*.sqlite", "*.dat", "*.xml",
    "*.properties", "*.config"
)
# ---------------------

foreach ($scanPath in $ScanPaths) {
    if (Test-Path $scanPath) {
        Write-Host "Scanning $scanPath for critical files..." -ForegroundColor Cyan

        # Get all matching files
        $foundFiles = Get-ChildItem -Path $scanPath -Include $Extensions -Recurse -ErrorAction SilentlyContinue

        foreach ($file in $foundFiles) {
            # 1. Determine the relative path (e.g., \Program Files\App\config.ini)
            # We strip the "D:" part from the FullName
            $relativePath = $file.FullName.Substring($SourceDrive.Length)

            # 2. Build the destination file path
            # Result: C:\backup\lynch\Program Files\App\config.ini
            $destFileFull = Join-Path -Path $BackupRoot -ChildPath $relativePath

            # 3. Get the destination FOLDER (Robocopy needs the folder, not the file name)
            $destFolder = Split-Path -Path $destFileFull -Parent

            # 4. Get just the file name for Robocopy
            $fileName = $file.Name

            # 5. Run Robocopy
            # Source folder is the parent of the found file
            $srcFolder = $file.DirectoryName

            Write-Host "Backing up: $fileName" -ForegroundColor Gray
            
            # /COPY:DAT  = Copy Data, Attributes, Timestamps
            # /R:0 /W:0  = No retries (fail fast if locked)
            # /NJH /NJS  = No Job Header/Summary (keeps output clean)
            # /NFL /NDL  = No File/Dir List (we already printed the name)
            # /XJ        = Exclude Junction points
            robocopy "$srcFolder" "$destFolder" "$fileName" /COPY:DAT /R:0 /W:0 /NJH /NJS /NFL /NDL /XJ
        }
    } else {
        Write-Warning "Path not found: $scanPath"
    }
}

Write-Host "`nBackup Complete." -ForegroundColor Green